/*============================================================================
  File:     Viewing Statistics Information.sql

  Summary:  See how statistics get automatically created based on query 
			scan. View sys.indexes,	use stats_date, use dbcc show_statistics 
			and use dbcc show_statistics to produce multiple tabular data sets.
  
  SQL Server Version: SQL Server 2005+
------------------------------------------------------------------------------
  Written by Kimberly L. Tripp, SYSolutions, Inc.

  For more scripts and sample code, check out 
    http://www.SQLskills.com

  This script is intended only as a supplement to demos and lectures
  given by Kimberly L. Tripp.  
  
  THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF 
  ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED 
  TO THE IMPLIED WARRANTIES OF MERCHA NTABILITY AND/OR FITNESS FOR A
  PARTICULAR PURPOSE.
============================================================================*/

-- These samples use the Credit database. You can download and restore the
-- credit database from here:
-- http://www.sqlskills.com/sql-server-resources/sql-server-demos/ 

USE [Credit];
GO

-- Just to make sure - set the DB to 120 (SQL 2014)
IF DATABASEPROPERTYEX('Credit', 'Collation') IS NOT NULL
	ALTER DATABASE Credit
		SET COMPATIBILITY_LEVEL = 120;
GO

--Here are the valid values for Compatibility Level
--80	SQL Server 2000
--90	SQL Server 2005
--100	SQL Server 2008 and SQL Server 2008 R2
--110	SQL Server 2012
--120	SQL Server 2014
--130	SQL Server 2016
--140	SQL Server 2017

-- Quick/general way to see index list
EXEC [sp_helpindex] 'dbo.member';
GO

-- You can get my version of sp_helpindex 
-- here: https://www.sqlskills.com/blogs/kimberly/category/sp_helpindex-rewrites/
EXEC [sp_SQLskills_helpindex] 'dbo.member';
GO

EXEC [sp_helpstats] 'dbo.member', 'all';
GO

CREATE INDEX [MemberName]
ON [dbo].[Member]([Lastname], [FirstName], [MiddleInitial]);
GO

DBCC SHOW_STATISTICS('Member', 'MemberName') 
GO

SET STATISTICS IO ON; --(turn on actual showplan)
GO

-----------------------------
-- READING THE DENSITY VECTOR
-----------------------------

-- If you want to see parameters and variables in procedures,
-- open/review the script: 01a ParametersVsVariablesInProcs.sql 

-- Using Density for LastName
SELECT 10000 * 0.03846154

-- Using Density for LastName, FirstName Combo
SELECT 10000 * 0.0001

-----------------------------
-- READING HISTOGRAMS
-----------------------------

-- Histogram Step Value
SELECT [m].*
FROM [dbo].[Member] AS [m]
WHERE [m].[LastName] = 'Chen' -- literals are KNOWN (sniffed)
GO

DBCC SHOW_STATISTICS('Member', 'MemberName') 
WITH HISTOGRAM;
GO

-- What if the data is not so even....
DBCC SHOW_STATISTICS ('Member', 'member_corporation_link')
WITH HISTOGRAM;
GO

-- Histogram Value In Step Range
SELECT [m].* 
FROM [dbo].[member] AS [m]
WHERE [m].[corp_no] = 404;
GO

-- Histogram Value In Step Range
SELECT [m].* 
FROM [dbo].[member] AS [m]
WHERE [m].[corp_no] = 405;
GO

-- Histogram Value In Step Range
SELECT [m].* 
FROM [dbo].[member] AS [m]
WHERE [m].[corp_no] = 406;
GO

-- Distinct Value Estimation
SELECT DISTINCT [m].[corp_no]
FROM [dbo].[member] AS [m];
GO

DBCC SHOW_STATISTICS('Member', 'member_corporation_link') 
WITH DENSITY_VECTOR;
GO

SELECT 1/0.0025;
GO

-- Unknown Values (not sniffed)
DECLARE @Lastname varchar(15) = 'Chen';
SELECT [m].*
FROM [dbo].[Member] AS [m]
WHERE [m].[LastName] = @Lastname;
GO

DBCC SHOW_STATISTICS('Member', 'MemberName') 
WITH DENSITY_VECTOR;
GO

-- Unknown Values (variable)
DECLARE @Lastname varchar(15) = 'Fish';
SELECT [m].*
FROM [dbo].[Member] AS [m]
WHERE [m].[LastName] = @Lastname;
GO

--Known/sniffed (literal)
SELECT [m].*
FROM [dbo].[Member] AS [m]
WHERE [m].[LastName] = 'Fish'; 
GO


-- So - what does this tell us about the relationship between LastNames and FirstNames?
EXEC [sp_helpindex] 'dbo.member'
EXEC [sp_helpstats] 'dbo.member', 'all';
GO -- look at messages window (Object does not have any stats outside of those on your indexes)

--We talked about this during the session so I wanted you to be able 
--to play with this example too!


-- What would you expect this query to do?
SET STATISTICS IO ON; --(turn on graphical showplan)
GO

SELECT [m].[LastName]
    , [m].[FirstName]
    , [m].[MiddleInitial]
	, [m].[Phone_no]
    , [m].[City]
FROM [dbo].[Member] AS [m]
WHERE [m].[FirstName] LIKE 'Kim%'
GO

-- Table Scan (always an option)
-- No Indexes to help find FIRSTNAMES...
-- What about scanning the NC index on LN,FN,MI and then doing bookmarks lookups...
-- seems risky? Would be good if we were guaranteed to only find a VERY small 
-- number of people with firstnames like 'Kim%'?

-- And yet it does - how did it know?

EXEC [sp_helpindex] 'dbo.member'
EXEC [sp_helpstats] 'dbo.member', 'all';
GO

DBCC SHOW_STATISTICS('Member', '_WA_Sys_00000003_0CBAE877')
WITH HISTOGRAM;
GO

-- Quick way to see stats_date for a specific table
EXEC [sp_autostats] N'dbo.Member';
GO

-- 2008R2 SP2+ or 2012 SP1+ and really useful for automation!
--sp_sqlskills_helpindex member
SELECT * 
FROM [sys].[dm_db_stats_properties]
(object_id('dbo.member'), 4);
GO

-- Seeing each tabular set from DBCC SHOW_STATISTICS 

DBCC SHOW_STATISTICS('[Credit].[dbo].[member]', 'MemberName')
WITH STAT_HEADER; 
GO

DBCC SHOW_STATISTICS('[Credit].[dbo].[member]', 'MemberName')
WITH DENSITY_VECTOR;
GO

DBCC SHOW_STATISTICS('Member', 'MemberName')
WITH HISTOGRAM;
GO


-- What about a generic "update" routine?
EXEC [sp_updatestats]; -- only requires one row to have changed...
GO

-- An even better choice - Ola's scripts:
-- http://ola.hallengren.com/